/*
 * Click nbfs://nbhost/SystemFileSystem/Templates/Licenses/license-default.txt to change this license
 * Click nbfs://nbhost/SystemFileSystem/Templates/Classes/Class.java to edit this template
 */
package org.milaifontanals.bdr;

import java.io.FileInputStream;
import java.sql.*;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.milaifontanals.empresa.Departament;

/**
 * Capa de persistència per la gestió de dels departaments i empleats de
 * l'empresa, segons el model definit en el projecte Empresa en un SGBDR
 *
 * L'aplicació que usi la capa de persistència ha de disposar d'un fitxer de
 * propietats (veure documentació dels constructors) accessible via load(Reader)
 * amb les propietats següents:<BR>
 * url : URL de connexió al SGBDR<BR>
 * user : Usuari<BR>
 * pw: Contrasenya<BR>
 *
 * @author Isidre Guixà
 */
public class CpJdbcEmpresa {

    private Connection c;

    private PreparedStatement psExistDept;
    private PreparedStatement psObtenirDept;
    private PreparedStatement psInserirDept;

    /**
     * Constructor que crea l'objecte capa de persistència, usant un fitxer de
     * configuració de nom empresaConfigJdbc.properties
     *
     * @throws CpJdbcEmpresaException si hi ha algun problema en el fitxer de
     * configuració o no s'estableix connexió
     */
    public CpJdbcEmpresa() throws CpJdbcEmpresaException {
        this("empresaConfigJdbc.properties");
    }

    /**
     * Constructor que crea l'objecte capa de persistència, usant un fitxer de
     * configuració de nom passat per paràmetre i, en cas de ser null o buit,
     * cerca fitxer de nom empresaConfigJdbc.properties
     *
     * @param nomFitxerPropietats
     * @throws CpJdbcEmpresaException si hi ha algun problema en el fitxer de
     * configuració o no s'estableix connexió
     */
    public CpJdbcEmpresa(String nomFitxerPropietats) throws CpJdbcEmpresaException {
        if (nomFitxerPropietats == null || nomFitxerPropietats.equals("")) {
            nomFitxerPropietats = "empresaConfigJdbc.properties";
        }
        Properties p = new Properties();
        try {
            p.load(new FileInputStream(nomFitxerPropietats));
        } catch (Exception ex) {
            throw new CpJdbcEmpresaException("Error en carregar fitxer de propietats " + nomFitxerPropietats, ex);
        }

        String url, user, pwd;
        url = p.getProperty("url");
        user = p.getProperty("user");
        pwd = p.getProperty("pwd");
        // Es podria fer comprovacions de NOT NULL

        c = null;
        try {
            c = DriverManager.getConnection(url, user, pwd);
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en establir connexió", ex);
        }
        try {
            c.setAutoCommit(false);
        } catch (SQLException ex) {
            try {
                c.close();
                c = null;
            } catch (SQLException ex1) {
            }
            throw new CpJdbcEmpresaException("Error en desactivar autocommit. Connexió no establerta.", ex);
        }
    }

    /**
     * Informa de l'existència de departament amb el codi indicat per paràmetre.
     *
     * @return Valor booleà que indica l'existència del departament indicat
     * @throws CpJdbcEmpresaException si es produeix algun error (introduir un
     * codi inadequat genera també aquesta excepció)
     */
    public boolean existeixDepartament(int codi) throws CpJdbcEmpresaException {
        if (codi <= 0 || codi > 99) {
            throw new CpJdbcEmpresaException("Intent de mirar existència de departament de codi " + codi + " no vàlid");
        }
        if (psExistDept == null) {
            try {
                psExistDept = c.prepareStatement("select 1 from dept where dept_no = ?");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psExistDept", ex);
            }
        }
        ResultSet rs = null;
        try {
            psExistDept.setInt(1, codi);
            rs = psExistDept.executeQuery();
            return rs.next();
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en comprovar existencia de departament", ex);
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
            } catch (SQLException ex) {
            }
        }
    }

    /**
     * Recupera objecte departament amb el codi indicat per paràmetre.
     *
     * @return Objecte Departament si existeix o null en cas contrari
     * @throws CpJdbcEmpresaException si es produeix algun error (introduir un
     * codi inadequat genera també aquesta excepció)
     */
    public Departament obtenirDepartament(int codi) throws CpJdbcEmpresaException {
        if (codi <= 0 || codi > 99) {
            throw new CpJdbcEmpresaException("Intent d'obtenir departament de codi " + codi + " no vàlid");
        }
        if (psObtenirDept == null) {
            try {
                psObtenirDept = c.prepareStatement("select dnom, loc from dept where dept_no = ?");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psObtenirDept", ex);
            }
        }
        ResultSet rs = null;
        try {
            psObtenirDept.setInt(1, codi);
            rs = psObtenirDept.executeQuery();
            if (!rs.next()) {
                return null;
            } else {
                return new Departament(codi, rs.getString("dnom"), rs.getString("loc"));
            }
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en comprovar existencia de departament", ex);
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
            } catch (SQLException ex) {
            }
        }

    }

    /**
     * Insereix objecte departament.
     *
     * @throws CpJdbcEmpresaException si es produeix algun error
     */
    public void inserirDepartament(Departament d) throws CpJdbcEmpresaException {
        if (psInserirDept == null) {
            try {
                psInserirDept = c.prepareStatement("insert into dept (dept_no, dnom, loc) values (?,?,?)");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psInserirDept", ex);
            }
        }
        Savepoint sp = null;
        try {
            psInserirDept.setInt(1, d.getCodi());
            psInserirDept.setString(2, d.getNom());
            psInserirDept.setString(3, d.getLocalitat());
            sp = c.setSavepoint();
            psInserirDept.executeUpdate();
        } catch (SQLException ex) {
            try {
                if (sp != null) {
                    c.rollback(sp);
                }
            } catch (SQLException ex1) {
            }
            throw new CpJdbcEmpresaException("Error en inserir departament " + d, ex);
        }

    }

    public void validarCanvis() throws CpJdbcEmpresaException {
        try {
            c.commit();
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en validar canvis", ex);
        }
    }

    public void desferCanvis() throws CpJdbcEmpresaException {
        try {
            c.rollback();
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en desfer canvis", ex);
        }
    }

    public void tancarCapa() throws CpJdbcEmpresaException {
        try {
            if (c != null) {
                c.rollback();
                c.close();
                c = null;
            }
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en tancar la connexió", ex);
        }
    }
}
