/*
 * Click nbfs://nbhost/SystemFileSystem/Templates/Licenses/license-default.txt to change this license
 * Click nbfs://nbhost/SystemFileSystem/Templates/Classes/Class.java to edit this template
 */
package org.milaifontanals.bdr;

import java.io.FileInputStream;
import java.sql.*;
import java.util.Calendar;
import java.util.Collection;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Properties;
import org.milaifontanals.empresa.Departament;
import org.milaifontanals.empresa.Empleat;

/**
 * Capa de persistència per la gestió de dels departaments i empleats de
 * l'empresa, segons el model definit en el projecte Empresa en un SGBDR
 *
 * L'aplicació que usi la capa de persistència ha de disposar d'un fitxer de
 * propietats (veure documentació dels constructors) accessible via load(Reader)
 * amb les propietats següents:<BR>
 * url : URL de connexió al SGBDR<BR>
 * user : Usuari<BR>
 * pw: Contrasenya<BR>
 *
 * @author Isidre Guixà
 */
public class CpJdbcEmpresa {

    private Connection c;

    private PreparedStatement psExistDept;
    private PreparedStatement psObtenirDept;
    private PreparedStatement psInserirDept;
    private PreparedStatement psExistEmp;
    private PreparedStatement psObtenirEmp;
    private PreparedStatement psInserirEmp;

    // Variables per guardar en memòria els objectes persistents:
    private HashMap<Integer, Departament> hmDept = new HashMap();
    private HashMap<Integer, Empleat> hmEmp = new HashMap();

    // variables per guardar en memòria els objectes enviats a la BD pendents de commit:
    private HashMap<Integer, Departament> hmDeptTemp = new HashMap();
    private HashMap<Integer, Empleat> hmEmpTemp = new HashMap();

    /**
     * Constructor que crea l'objecte capa de persistència, usant un fitxer de
     * configuració de nom empresaConfigJdbc.properties
     *
     * @throws CpJdbcEmpresaException si hi ha algun problema en el fitxer de
     * configuració o no s'estableix connexió
     */
    public CpJdbcEmpresa() throws CpJdbcEmpresaException {
        this("empresaConfigJdbc.properties");
    }

    /**
     * Constructor que crea l'objecte capa de persistència, usant un fitxer de
     * configuració de nom passat per paràmetre i, en cas de ser null o buit,
     * cerca fitxer de nom empresaConfigJdbc.properties
     *
     * @param nomFitxerPropietats
     * @throws CpJdbcEmpresaException si hi ha algun problema en el fitxer de
     * configuració o no s'estableix connexió
     */
    public CpJdbcEmpresa(String nomFitxerPropietats) throws CpJdbcEmpresaException {
        if (nomFitxerPropietats == null || nomFitxerPropietats.equals("")) {
            nomFitxerPropietats = "empresaConfigJdbc.properties";
        }
        Properties p = new Properties();
        try {
            p.load(new FileInputStream(nomFitxerPropietats));
        } catch (Exception ex) {
            throw new CpJdbcEmpresaException("Error en carregar fitxer de propietats " + nomFitxerPropietats, ex);
        }

        String url, user, pwd;
        url = p.getProperty("url");
        user = p.getProperty("user");
        pwd = p.getProperty("pwd");
        // Es podria fer comprovacions de NOT NULL

        c = null;
        try {
            c = DriverManager.getConnection(url, user, pwd);
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en establir connexió", ex);
        }
        try {
            c.setAutoCommit(false);
        } catch (SQLException ex) {
            try {
                c.close();
                c = null;
            } catch (SQLException ex1) {
            }
            throw new CpJdbcEmpresaException("Error en desactivar autocommit. Connexió no establerta.", ex);
        }
    }

    /**
     * Informa de l'existència de departament amb el codi indicat per paràmetre.
     *
     * @param codi Codi del departament pel que es vol comprovar existència
     * @return Valor booleà que indica l'existència del departament indicat
     * @throws CpJdbcEmpresaException si es produeix algun error (introduir un
     * codi inadequat genera també aquesta excepció)
     */
    public boolean existeixDepartament(int codi) throws CpJdbcEmpresaException {
        if (codi <= 0 || codi > 99) {
            throw new CpJdbcEmpresaException("Intent de mirar existència de departament de codi " + codi + " no vàlid");
        }
        if (hmDept.containsKey(codi) || hmDeptTemp.containsKey(codi)) {
            return true;
        }
        if (psExistDept == null) {
            try {
                psExistDept = c.prepareStatement("select 1 from dept where dept_no = ?");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psExistDept", ex);
            }
        }
        ResultSet rs = null;
        try {
            psExistDept.setInt(1, codi);
            rs = psExistDept.executeQuery();
            return rs.next();
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en comprovar existencia de departament de codi " + codi, ex);
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
            } catch (SQLException ex) {
            }
        }
    }

    /**
     * Informa de l'existència d'empleat amb el codi indicat per paràmetre.
     *
     * @param codi Codi de l'empleat pel que es vol comprovar existència
     * @return Valor booleà que indica l'existència del departament indicat
     * @throws CpJdbcEmpresaException si es produeix algun error (introduir un
     * codi inadequat genera també aquesta excepció)
     */
    public boolean existeixEmpleat(int codi) throws CpJdbcEmpresaException {
        if (codi <= 0 || codi > 9999) {
            throw new CpJdbcEmpresaException("Intent de mirar existència d'empleat de codi " + codi + " no vàlid");
        }
        if (hmEmp.containsKey(codi) || hmEmpTemp.containsKey(codi)) {
            return true;
        }
        if (psExistEmp == null) {
            try {
                psExistEmp = c.prepareStatement("select 1 from emp where emp_no = ?");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psExistEmp", ex);
            }
        }
        ResultSet rs = null;
        try {
            psExistEmp.setInt(1, codi);
            rs = psExistEmp.executeQuery();
            return rs.next();
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en comprovar existencia d'empleat de codi " + codi, ex);
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
            } catch (SQLException ex) {
            }
        }
    }

    /**
     * Recupera objecte departament amb el codi indicat per paràmetre.
     *
     * @param codi Codi del departament que es vol recuperar
     * @return Objecte Departament si existeix o null en cas contrari
     * @throws CpJdbcEmpresaException si es produeix algun error (introduir un
     * codi inadequat genera també aquesta excepció)
     */
    public Departament obtenirDepartament(int codi) throws CpJdbcEmpresaException {
        if (codi <= 0 || codi > 99) {
            throw new CpJdbcEmpresaException("Intent d'obtenir departament de codi " + codi + " no vàlid");
        }
        if (hmDept.containsKey(codi)) {
            return (Departament) hmDept.get(codi);
        }
        if (hmDeptTemp.containsKey(codi)) {
            return (Departament) hmDeptTemp.get(codi);
        }
        if (psObtenirDept == null) {
            try {
                psObtenirDept = c.prepareStatement("select dnom, loc from dept where dept_no = ?");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psObtenirDept", ex);
            }
        }
        ResultSet rs = null;
        try {
            psObtenirDept.setInt(1, codi);
            rs = psObtenirDept.executeQuery();
            if (!rs.next()) {
                return null;
            } else {
                Departament d = new Departament(codi, rs.getString("dnom"), rs.getString("loc"));
                hmDept.put(codi, d);
                return d;
            }
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en recuperar departament de codi " + codi, ex);
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
            } catch (SQLException ex) {
            }
        }

    }

    /**
     * Recupera objecte Empleat amb el codi indicat per paràmetre.
     *
     * @param codi Codi de l'empleat que es vol recuperar
     * @return Objecte Empleat si existeix o null en cas contrari
     * @throws CpJdbcEmpresaException si es produeix algun error (introduir un
     * codi inadequat genera també aquesta excepció)
     */
    public Empleat obtenirEmpleat(int codi) throws CpJdbcEmpresaException {
        if (codi <= 0 || codi > 9999) {
            throw new CpJdbcEmpresaException("Intent d'obtenir empleat de codi " + codi + " no vàlid");
        }
        if (hmEmp.containsKey(codi)) {
            return (Empleat) hmEmp.get(codi);
        }
        if (hmEmpTemp.containsKey(codi)) {
            return (Empleat) hmEmpTemp.get(codi);
        }
        if (psObtenirEmp == null) {
            try {
                psObtenirEmp = c.prepareStatement("select cognom, ofici, cap, data_alta, salari, comissio, dept_no "
                        + "from emp where emp_no = ?");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psObtenirEmp", ex);
            }
        }
        ResultSet rs = null;
        try {
            psObtenirEmp.setInt(1, codi);
            rs = psObtenirEmp.executeQuery();
            if (!rs.next()) {
                return null;
            } else {
                Departament dept = null;
                Integer codiDept = rs.getInt("dept_no");
                if (!rs.wasNull()) {
                    dept = this.obtenirDepartament(codiDept);
                }

                Double salari = rs.getDouble("salari");
                if (rs.wasNull()) {
                    salari = null;
                }
                Double comissio = rs.getDouble("comissio");
                if (rs.wasNull()) {
                    comissio = null;
                }
                Date auxDataAlta = rs.getDate("data_alta");
                Calendar data_alta = null;
                if (auxDataAlta != null) {  // O també: !rs.wasNull()
                    data_alta = new GregorianCalendar();
                    data_alta.setTime(auxDataAlta);
                }

                Empleat e = new Empleat(codi, rs.getString("cognom"),
                        rs.getString("ofici"),
                        data_alta, salari, comissio, null, dept);

                // Deixem la recuperació del cap per la última operació a efectuar amb el ResultSet rs
                // doncs si la crida a obtenirEmpleat provoca nou executeUpdate sobre psObtenirEmpleat,
                // com que aquest és global, el ResultSet rs d'aquí quedaria tancat i no podríem consultar-lo
                Empleat cap = null;
                Integer codiCap = rs.getInt("cap");
                if (!rs.wasNull()) {
                    e.setCap(this.obtenirEmpleat(codiCap));
                }

                hmEmp.put(codi, e);
                return e;
            }
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en recuperar empleat de codi " + codi, ex);
        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
            } catch (SQLException ex) {
            }
        }

    }

    /**
     * Insereix objecte departament.
     *
     * @param d Departament a inserir
     * @throws CpJdbcEmpresaException si es produeix algun error.
     */
    public void inserirDepartament(Departament d) throws CpJdbcEmpresaException {
        if (psInserirDept == null) {
            try {
                psInserirDept = c.prepareStatement("insert into dept (dept_no, dnom, loc) values (?,?,?)");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psInserirDept", ex);
            }
        }
        Savepoint sp = null;
        try {
            psInserirDept.setInt(1, d.getCodi());
            psInserirDept.setString(2, d.getNom());
            psInserirDept.setString(3, d.getLocalitat());
            sp = c.setSavepoint();
            psInserirDept.executeUpdate();
            hmDeptTemp.put(d.getCodi(), d);
        } catch (SQLException ex) {
            try {
                if (sp != null) {
                    c.rollback(sp);
                }
            } catch (SQLException ex1) {
            }
            throw new CpJdbcEmpresaException("Error en inserir departament " + d, ex);
        }

    }

    /**
     * Insereix objecte Empleat.
     *
     * @param e Empleat a inserir
     * @throws CpJdbcEmpresaException si es produeix algun error. Si l'empleat
     * té cap i/ol departament que no són a la BD, genera error.
     */
    public void inserirEmpleat(Empleat e) throws CpJdbcEmpresaException {
        if (psInserirEmp == null) {
            try {
                psInserirEmp = c.prepareStatement("insert into emp (emp_no, cognom, ofici, cap, data_alta, salari, comissio, dept_no)"
                        + " values (?,?,?,?,?,?,?,?)");
            } catch (SQLException ex) {
                throw new CpJdbcEmpresaException("Error en preparar statement psInserirEmp", ex);
            }
        }
        Savepoint sp = null;
        try {
            psInserirEmp.setInt(1, e.getCodi());
            psInserirEmp.setString(2, e.getCognom());
            psInserirEmp.setString(3, e.getOfici());
            if (e.getCap() != null) {
                psInserirEmp.setInt(4, e.getCap().getCodi());
            } else {
                psInserirEmp.setNull(4, java.sql.Types.NULL);
            }
            if (e.getDataAlta() != null) {
                // Alerta! Els objectes Date en JDBC són java.sql.Date derivada de java.util.Date
                // Per tant, el mètode setDate sobre un PreparedStatement ha de rebre un 
                // objecte java.sql.Date i no pas java.util.Date. 
                // El mètode e.getDataAlta() retorna un objecte Calendar i cal
                // espavilar per obtenir el corresponent java.sql.Date
                java.sql.Date aux = new java.sql.Date(e.getDataAlta().getTimeInMillis());
                psInserirEmp.setDate(5, aux);
                // O en una sola línia:
//                psInserirEmp.setDate(5, new java.sql.Date(e.getDataAlta().getTimeInMillis()));
                // Hem invocat el constructor de java.sql.Date a partir del valor de
                // l'objecte Calendar en milisegons. No calia introduir el nom del 
                // paquet per que no tanim import java.util.Date, però quan en el 
                // codi conviuen classes amb igual nom, cal usar el nom del paquet.
            } else {
//                psInserirEmp.setNull(5, java.sql.Types.NULL);
//                En aquest cas també pot ser:
                psInserirEmp.setDate(5, null);
            }
            if (e.getSalari() != null) {
                psInserirEmp.setDouble(6, e.getSalari());
            } else {
                psInserirEmp.setNull(6, java.sql.Types.NULL);
            }
            if (e.getComissio() != null) {
                psInserirEmp.setDouble(7, e.getComissio());
            } else {
                psInserirEmp.setNull(7, java.sql.Types.NULL);
            }
            if (e.getDept() != null) {
                psInserirEmp.setInt(8, e.getDept().getCodi());
            } else {
                psInserirEmp.setNull(8, java.sql.Types.NULL);
            }
            sp = c.setSavepoint();
            psInserirEmp.executeUpdate();
            hmEmpTemp.put(e.getCodi(), e);
        } catch (SQLException ex) {
            try {
                if (sp != null) {
                    c.rollback(sp);
                }
            } catch (SQLException ex1) {
            }
            throw new CpJdbcEmpresaException("Error en inserir empleat " + e, ex);
        }

    }

    public void validarCanvis() throws CpJdbcEmpresaException {
        try {
            c.commit();
            Collection<Departament> cDept = hmDeptTemp.values();
            for (Departament d : cDept) {
                hmDept.put(d.getCodi(), d);
            }
            hmDeptTemp = new HashMap();
            Collection<Empleat> cEmp = hmEmpTemp.values();
            for (Empleat e : cEmp) {
                hmEmp.put(e.getCodi(), e);
            }
            hmEmpTemp = new HashMap();
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en validar canvis", ex);
        }
    }

    public void desferCanvis() throws CpJdbcEmpresaException {
        try {
            c.rollback();
            hmDeptTemp = new HashMap();
            hmEmpTemp = new HashMap();
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en desfer canvis", ex);
        }
    }

    public void tancarCapa() throws CpJdbcEmpresaException {
        try {
            if (c != null) {
                c.rollback();
                c.close();
                c = null;
            }
        } catch (SQLException ex) {
            throw new CpJdbcEmpresaException("Error en tancar la connexió", ex);
        }
    }
}
